import { prisma } from "../database/prisma.client";
import { IAuthRepository, IResetPassword, IUser, IUserWithoutPassword, UserCreate } from "../interfaces/auth.interfaces";
import bcrypt from "bcrypt";
import jwt from "jsonwebtoken";
import crypto from "crypto";


class AuthRepositoryPrisma implements IAuthRepository {
    async create(data: UserCreate): Promise<IUser> {
        const hashedPassword = await bcrypt.hash(data.password, 10);
        const result = await prisma.user.create({
            data: {
                email: data.email,
                password: hashedPassword,
                role: data.role,
                cliente: data.cliente ? {
                    create: data.cliente
                } : undefined,
                profissional: data.profissional ? {
                    create: data.profissional
                } : undefined
            },
            include: {
                cliente: true,
                profissional: true
            }
        });
        const { password, ...userWithoutPassword } = result;
        return userWithoutPassword as never;
    }

    async login(data: UserCreate): Promise<IUserWithoutPassword> {
        const user = await prisma.user.findUnique({
            where: {
                email: data.email
            }
        });

        if (!user) {
            throw new Error('Usuário não encontrado');
        }

        const isPasswordValid = await bcrypt.compare(data.password, user.password);
        if (!isPasswordValid) {
            throw new Error('Senha incorreta');
        }

        const { password, ...userWithoutPassword } = user;
        const token = jwt.sign({ id: user.id }, process.env.JWT_PASS ?? "", { expiresIn: '1h' });


        return { ...userWithoutPassword, token };
    };
    async forgotPassword(email: string): Promise<void> {
        const user = await prisma.user.findUnique({
            where: {
                email
            }
        });
        if (!user) {
            throw new Error('Usuário não encontrado');
        }
        const token = crypto.randomBytes(8).toString("hex");
        const now = new Date();
        now.setHours(now.getHours() + 1);
        const existingToken = await prisma.authToken.findFirst({
            where: {
                userId: user.id,
            },
        });
        if (existingToken) {
            // Token exists, update it
            await prisma.authToken.update({
                where: {
                    id: existingToken.id,
                },
                data: {
                    passwordResetToken: token,
                    passwordResetExpires: now,
                },
            });
        } else {
            // Token does not exist, create it
            await prisma.authToken.create({
                data: {
                    userId: user.id,
                    passwordResetToken: token,
                    passwordResetExpires: now,
                },
            });
        }
        /*
        enviarEmail(user.email, "Reset Password ", 'esqueciasenha', {
            dados: {
                nome: user.name,
                conteudo: `http://localhost:3000/reset-password?token=${token}`,
                assunto: "Reset Password ",
                operatingSystem: "",
                browserName: "",
            }
        });
        */
    }

    async resetPassword(data: IResetPassword): Promise<void> {
        const { token, password, email } = data;
        const tokenData = await prisma.authToken.findFirst({
            where: {
                passwordResetToken: token,
            },
        });
        if (!tokenData) {
            throw new Error('Token inválido');
        }
        const user = await prisma.user.findUnique({
            where: {
                id: tokenData.userId,
            },
        });
        if (!user) {
            throw new Error('Usuário não encontrado');
        }
        const hashedPassword = await bcrypt.hash(password, 10);
        await prisma.user.update({
            where: {
                id: user.id,
            },
            data: {
                password: hashedPassword,
            },
        });
    }

}


export { AuthRepositoryPrisma }